//
// 64drive USB loader
// Copyright (c) 2011-2012 Marshall H / Retroactive
// http://64drive.retroactive.be
//
// You're welcome to port to other platforms. Send me a link or zip file.
// Mess with the firmware/bootloader upgrading at your own risk.
//
// Uses FTDI D2XX drivers. http://www.ftdichip.com/Drivers/D2XX.htm
//
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// The above copyright notice and this permission notice shall be included in all 
// copies or substantial portions of the Software.
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
// IN THE SOFTWARE.

// Changes : November 12, 2012
// - Modified VS2008 project to support static linking of the VCRT
// - Removed precompiled headers
// - Removed unicode settings
// - Compilable as strictly C
// - Added MIT license
// Next firmware revision will carry a significant number of changes, and 
// this code will probably be superceded. Don't worry, the code is much better
// than this :)

#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x0500
#endif

#include <stdio.h>
#include <tchar.h>
#include <windows.h>
#pragma comment(lib, "FTD2XX.lib")
#ifndef DEBUG
#define FTD2XX_EXPORTS	// used for statically linking FTDI
#endif
#include "ftd2xx.h"

#define CHUNK_SIZE (512*1024)
unsigned char buffer[CHUNK_SIZE];

int		save_type = 0;
int		save_num = 5;
char	*save_type_list[] = {	"None (default)",
								"EEPROM 4k", 
								"EEPROM 16k", 
								"SRAM 256kbit", 
								"FlashRAM 1Mbit", 
								"SRAM 768Kbit (for Dezaemon 3D)"};

char	*compile_date = __TIMESTAMP__;
long	sdram_addr = 0;

unsigned int file_size = 0;
char image_filename[2048];


void fail(FT_STATUS st);
void load_image(char *filename);
int swap_endian(int val);
void write_header(char* tx_buffer, unsigned char operation, int sdram_start, int num_words);
void write_header_bus(char* tx_buffer, int address, unsigned short data);
void write_header_bus_long(char* tx_buffer, int address, unsigned short data);



int swap_endian(int val)
{
	return ((val<<24) ) | 
		   ((val<<8)  & 0x00ff0000) |
		   ((val>>8)  & 0x0000ff00) | 
		   ((val>>24) );
}

void write_header_bus(char* tx_buffer, int address, unsigned short data)
{
	tx_buffer[0] = 0x90; // operation
	tx_buffer[1] = 0x08; // length of packet

	*(int *)&tx_buffer[2] = swap_endian(address);
	tx_buffer[6] = data & 0xff;
	tx_buffer[7] = (data >> 8) & 0xff;
}

void write_header_bus_long(char* tx_buffer, int address, unsigned short data)
{
	tx_buffer[0] = 0x91; // operation
	tx_buffer[1] = 0x08; // length of packet

	*(int *)&tx_buffer[2] = swap_endian(address);
	tx_buffer[6] = data & 0xff;
	tx_buffer[7] = (data >> 8) & 0xff;
}

void write_header(char* tx_buffer, unsigned char operation, int sdram_start, int num_words)
{
	memset(tx_buffer, 0, 32);
	tx_buffer[0] = operation; // operation
	tx_buffer[1] = 0x0A; // length of packet

	*(int *)&tx_buffer[2] = swap_endian(num_words);
	*(int *)&tx_buffer[6] = swap_endian(sdram_start);
}


#define MODE_UNSPECIFIED	0
#define MODE_LOADIMAGE		1
#define MODE_DUMPRAM		2
#define MODE_SETSAVE		4

#define MODE_BOOTLOADER		16
#define MODE_FPGACONFIG		32

#define ERR_TIMEOUT			9
#define ERR_CANTOPENDEVICE	10
#define ERR_CANTOPENIMAGE	11
#define ERR_INVALIDSAVE		12
#define ERR_INVALIDBTLDR	13
#define ERR_INVALIDFIRM		14

int main(int argc, char* argv[])
{
	int i;
	int last_option;
	int mode = MODE_UNSPECIFIED;
	DWORD ft_devices;
	DWORD found = 8;
	FT_STATUS status;
	FT_DEVICE_LIST_INFO_NODE *devInfo;
	FT_HANDLE handle;

	DWORD bytes_written;
	FILE* fp;
	char *filename_start;
	char tx_buffer[64];

	unsigned char *bootld_ptr;

	int blocks_done;
	int blocks_left;
	LONGLONG llTimeDiff;
	double dftDuration;

	printf("\n");
	printf(" 64drive USB Loader\n----------------------\n");
	printf(" (c) 2011-2012 Retroactive\n");
	printf(" Compiled %s\n", compile_date);
	for (i = 1; i < argc; ++i)  {
		if (argv[i][0] == '-'){ // looks like an argument 
       
			switch(argv[i][1]){ // look at next char
            
                case 'w': 
					load_image(image_filename);
					mode |= MODE_LOADIMAGE;
                    break;

                default: 
                    break;
            }
            last_option = argv[i][1];
        }else{
            // this is not a -option argument, so it must be a textual argument        
            switch(last_option) 
            {   
				case 'a':
					sscanf(argv[i], "%x", &sdram_addr);
					break;
                case 'l': 
                    strcpy(image_filename, argv[i]);
					mode |= MODE_LOADIMAGE;
                    break;
				case 'd':
					strcpy(image_filename, argv[i]);
					sscanf(argv[i+1], "%x", &file_size);
					mode |= MODE_DUMPRAM;
					break;
				case 's':
					sscanf(argv[i], "%d", &save_type);
					mode |= MODE_SETSAVE;
					break;
                case 'b': 
                    strcpy(image_filename, argv[i]);
					mode = MODE_BOOTLOADER;
					sdram_addr = 0;
                    break;
                case 'f':
					strcpy(image_filename, argv[i]);
					mode = MODE_FPGACONFIG;
					sdram_addr = 0;
					break;
				
				
                default: 
                    break;
            }
            last_option = '\0';
        }
    }

	if(mode == MODE_UNSPECIFIED){
        // filename was not given, complain
        printf(" - Invalid parameter(s)\n - Parameters:\n");
		printf("    -w\t\t\t\t Show dialog to open file\n");
		printf("    -l <filename.z64>\t\t Load image to offset in SDRAM\n");
		printf("    -d <filename.bin> <bytes>\t Dump to file, from SDRAM\n");
		printf("    -a <offset in hex>\t\t Set SDRAM offset (default is 0x0)\n");
		printf("    -s <save type>\t\t Set save type \n");
		for(i = 0; i < save_num+1; i++){
			printf("    \t\t\t\t   %d - %s\n", i, save_type_list[i]);
		}

		printf(" - Update commands\n");
		printf("    -b <filename.bin>\t\t Update bootloader\n");
		printf("    -f <filename.bin>\t\t Update firmware (danger of bricking)\n");
        exit(1);
    }
	
	

	////////////////////////////////////////////////////
	// enumerate
	//
	status = FT_CreateDeviceInfoList(&ft_devices);
	if(status != FT_OK) fail(status);
	if(ft_devices == 0){
		printf(" - No devices found.\n");
		exit(-1);
	}else{
		// allocate storage
		devInfo = (FT_DEVICE_LIST_INFO_NODE*)malloc(sizeof(FT_DEVICE_LIST_INFO_NODE)*ft_devices);
		// get device info list
		status = FT_GetDeviceInfoList(devInfo, &ft_devices);
		if(status == FT_OK){
			/*
			for(i = 0; i < ft_devices; i++){
				printf("    [Device %d]\n", i);
				printf("     -- Flags\t\t0x%x\n",devInfo[i].Flags); 
				printf("     -- Type\t\t0x%x\n",devInfo[i].Type); 
				printf("     -- ID\t\t0x%x\n",devInfo[i].ID); 
				printf("     -- LocId\t\t0x%x\n",devInfo[i].LocId); 
				printf("     -- SerialNumber\t%s\n",devInfo[i].SerialNumber); 
				printf("\t%s\n",devInfo[i].Description); 
			} */
		}
	}

	////////////////////////////////////////////////////
	// look for device in list
	//
	for(i = 0; i < ft_devices; i++){
		if(strcmp(devInfo[i].Description, "64drive USB device A") == 0){
			printf(" - Found 64drive at 0x%x\n", devInfo[i].LocId);
			found = 0;
			break;
		}
	}
	free(devInfo);
	
	if(found != FT_OK){
		printf(" - No attached devices are 64drives\n");
		exit(-1);
	}

	////////////////////////////////////////////////////
	// open device
	//
	//printf(" - Opening device... ");
	status = FT_Open(0, &handle);
	if(status == FT_OK){
		//printf("OK!\n");
	}else{
		printf(" *** Couldn't open device.\n");
		fail(ERR_CANTOPENDEVICE);
	}
	// set read/write timeouts
	fail(FT_SetTimeouts(handle, 4000, 4000));
	fail(FT_ResetDevice(handle));




	////////////////////////////////////////////////////
	// open image file
	//
	filename_start = strrchr(image_filename, '\\');
	if(filename_start == NULL) filename_start = image_filename-1;
	printf(" - Opening file %s ", filename_start+1);
	fp = fopen(image_filename, mode&MODE_DUMPRAM ? "wb" : "rb");
	if(fp == NULL) {
		printf("\n\n *** Couldn't open file\n");
		fail(ERR_CANTOPENIMAGE);
	}

	fseek(fp, 0, SEEK_END);
	if(!(mode&MODE_DUMPRAM))file_size = ftell(fp);
	fseek(fp, 0, SEEK_SET);

	printf("(%d bytes)\n", file_size);

	/////////////////////////////////////////////////////
	// set save type
	//
	if(mode & MODE_SETSAVE){
		if(save_type < 0 || save_type > save_num){
			printf("\n *** Invalid save specified (%d)\n", save_type);
			fail(ERR_INVALIDSAVE);
		}
		printf(" - Setting save type to %s\n", save_type_list[save_type]);
		memset(tx_buffer, 0, 32);
		tx_buffer[0] = 0xd0;	// operation
		tx_buffer[1] = 4;		// length of packet
		tx_buffer[2] = save_type;
		fail(FT_Write(handle, tx_buffer, 4, &bytes_written));
	}
	

	/////////////////////////////////////////////////////
	// bulk erase bootloader
	//
	if(mode & MODE_BOOTLOADER){
		fread(tx_buffer, 16, 1, fp);
		if(strncmp(tx_buffer, "RETROACTIVE_2011", 16) != 0){
			printf(" * Invalid header. Make sure you have the right file!\n");
			fail(ERR_INVALIDBTLDR);
		}
		printf("\n\n");
		printf(" * Warning: about to erase and rewrite your bootloader!!!\n");
		printf(" * Be sure you have a valid file.\n");
		printf(" * Continue? (y/n) ");
		if(getchar() != 'y') fail(MODE_BOOTLOADER);
		printf("\n * Please ensure your cartridge is not in the N64!\n");
		printf(" * Press enter to continue.\n");
		getchar();getchar();

		if(file_size != 524288+16){
			printf(" * Bootloader image is not the proper size\n");
			fail(MODE_BOOTLOADER);
		}

		printf(" - Bulk erasing bootloader\n");
			
		memset(tx_buffer, 0, 32);

		// switch cpld to whole-chip access
		write_header_bus(tx_buffer, 0x17000000, 0x0000);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));	

		write_header_bus(tx_buffer, 0x5555*2, 0xAA00);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));
		write_header_bus(tx_buffer, 0x2AAA*2, 0x5500);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));
		write_header_bus(tx_buffer, 0x5555*2, 0x8000);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));

		write_header_bus(tx_buffer, 0x5555*2, 0xAA00);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));
		write_header_bus(tx_buffer, 0x2AAA*2, 0x5500);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));
		write_header_bus(tx_buffer, 0x5555*2, 0x1000);
		fail(FT_Write(handle, tx_buffer, 8, &bytes_written));
		Sleep(1000);
	}
	

	
	/////////////////////////////////////////////////////
	// write bootloader
	//
	if(mode & MODE_BOOTLOADER){
		int bytes_left = file_size;
		int bytes_done = 0;
		int bytes_do;
		unsigned short word;

		file_size = 524288;
		sdram_addr = 0;
		printf(" - Writing bootloader\n");
		while(1){
			if(bytes_left >= CHUNK_SIZE/32) 
				bytes_do = CHUNK_SIZE/32;
			else
				bytes_do = bytes_left;
			if(bytes_do <= 0) break;

			bootld_ptr = &buffer[0];

			memset(bootld_ptr, 0, CHUNK_SIZE);

			for(i = 0; i < bytes_do/2; i++){
				fread(&word, 2, 1, fp);
				write_header_bus((char*)bootld_ptr,	0x5555*2, 0xAA00);
				write_header_bus((char*)bootld_ptr+8,	0x2AAA*2, 0x5500);
				write_header_bus((char*)bootld_ptr+16,	0x5555*2, 0xA000);
				write_header_bus_long((char*)bootld_ptr+24, sdram_addr, word);
				sdram_addr += 2;
				bootld_ptr += 32;
			}
			fail(FT_Write(handle, buffer, CHUNK_SIZE, &bytes_written));

			bytes_left -= bytes_do;
			bytes_done += bytes_do;
			//sdram_addr += bytes_do;

			// progress bar
			blocks_done = ((float)bytes_done / file_size)*64;
			blocks_left = 64-blocks_done;
			printf("\r   ");
			for(i = 0; i < blocks_done; i++) printf("\xdb");
			for(i = 0; i < blocks_left; i++) printf("\xb0");
		}
		printf("\n - Success!\n");
		printf(" - Please unplug the USB cable and plug it back in.\n");
	}
	
	
	/////////////////////////////////////////////////////
	// bulk erase config
	//
	
	if(mode & MODE_FPGACONFIG){
		fread(tx_buffer, 16, 1, fp);
		if(strncmp(tx_buffer, "RETROACTIVE-2011", 16) != 0){
			printf(" * Invalid header. Make sure you have the right file!\n");
			fail(ERR_INVALIDFIRM);
		}

		printf("\n **********************************************************\n");
		printf(" * Warning: about to erase and rewrite your firmware!!!\n");
		printf(" * If this fails for any reason (power goes out, invalid file)\n");
		printf(" * your 64drive will be rendered INOPERABLE!\n");
		printf(" * \n");
		printf(" * This must be the first operation performed right after\n");
		printf(" * plugging in USB, and within 10 seconds or it is locked out.\n");
		printf(" * Continue? (y/n) ");
		if(getchar() != 'y') fail(MODE_FPGACONFIG);

		printf("\n - Bulk erasing config (9 seconds)\n");
		memset(tx_buffer, 0, 32);
		tx_buffer[0] = 0x80;	// operation
		tx_buffer[1] = 4;		// length of packet
		tx_buffer[2] = 0xBE;	// magic!
		tx_buffer[3] = 0xEF;
		fail(FT_Write(handle, tx_buffer, 4, &bytes_written));
		Sleep(9000);
	}
	
	/////////////////////////////////////////////////////
	// write config
	//
	if(mode & MODE_FPGACONFIG){
		int bytes_left = file_size;
		int bytes_done = 0;
		int bytes_do;

		printf(" - Writing FPGA config\n");
		while(1){
			if(bytes_left >= 1024) 
				bytes_do = 1024;
			else
				bytes_do = bytes_left;
			if(bytes_do <= 0) break;
			
			write_header(tx_buffer, 0x70, sdram_addr, bytes_do);
			fail(FT_Write(handle, tx_buffer, 10, &bytes_written));

			fread(buffer, bytes_do, 1, fp);
			fail(FT_Write(handle, buffer, bytes_do, &bytes_written));

			// check for a timeout
			if(bytes_written == 0){
				printf("\n *** Timed out\n");
				printf(" *** Please cycle power and try again\n");
				fail(ERR_TIMEOUT);
			}

			bytes_left -= bytes_written;
			bytes_done += bytes_written;
			sdram_addr += bytes_written;

			// progress bar
			blocks_done = ((float)bytes_done / file_size)*64;
			blocks_left = 64-blocks_done;
			printf("\r   ");
			for(i = 0; i < blocks_done; i++) printf("\xdb");
			for(i = 0; i < blocks_left; i++) printf("\xb0");
		}
		printf("\n - Success!\n");
		printf(" - Please disconnect the USB cable and plug it back in.\n");
	}

	
	
	/////////////////////////////////////////////////////
	// start transfer
	//
	if(mode & MODE_LOADIMAGE){
		int bytes_left = file_size;
		int bytes_done = 0;
		int bytes_do;

		LARGE_INTEGER liStart;
		LARGE_INTEGER liStop;
		LARGE_INTEGER Frequency;
		QueryPerformanceFrequency(&Frequency);
		QueryPerformanceCounter(&liStart);

		printf(" - Writing to SDRAM at 0x%X \n", sdram_addr);
		while(1){
			if(bytes_left >= CHUNK_SIZE) 
				bytes_do = CHUNK_SIZE;
			else
				bytes_do = bytes_left;
			if(bytes_do <= 0) break;
			
			write_header(tx_buffer, 0x10, sdram_addr/2, bytes_do/2);
			status = FT_Write(handle, tx_buffer, 10, &bytes_written);
			if(status == FT_OK){
				//printf("    Wrote [header] %d bytes ok\n", bytes_written);
			}else{
				fail(MODE_LOADIMAGE);
			}

			fread(buffer, bytes_do, 1, fp);
			status = FT_Write( handle, buffer, bytes_do, &bytes_written);
			if(status == FT_OK){
				//printf("    Wrote [data]   %d bytes ok\n", bytes_written);
			}else{
				fail(MODE_LOADIMAGE);
			}

			// check for a timeout
			if(bytes_written == 0){
				printf("\n *** Timed out\n");
				printf(" *** Please cycle power and try again\n");
				fail(ERR_TIMEOUT);
			}

			bytes_left -= bytes_written;
			bytes_done += bytes_written;
			sdram_addr += bytes_written;

			// progress bar
			blocks_done = ((float)bytes_done / file_size)*64;
			blocks_left = 64-blocks_done;
			printf("\r   ");
			for(i = 0; i < blocks_done; i++) printf("\xdb");
			for(i = 0; i < blocks_left; i++) printf("\xb0");
		}

		// stop the timee
		QueryPerformanceCounter(&liStop);
		llTimeDiff = liStop.QuadPart - liStart.QuadPart;
		// get the difference of the timer
		dftDuration = (double) llTimeDiff * 1000.0 / (double) Frequency.QuadPart;
		// erase progress bar
		printf("\r");	for(i = 0; i < 79; i++) printf(" "); printf("\r");
		printf(" - Completed in %.2f seconds (%.2f MB/sec)\n", dftDuration/1000, file_size/1024/1024/(dftDuration/1000));
		
	}


	/////////////////////////////////////////////////////
	// start transfer
	//
	if(mode & MODE_DUMPRAM){
		int bytes_left = file_size;
		int bytes_done = 0;
		int bytes_do;


		printf(" - Dumping from SDRAM at 0x%X \n", sdram_addr);
		while(1){
			if(bytes_left >= CHUNK_SIZE) 
				bytes_do = CHUNK_SIZE;
			else
				bytes_do = bytes_left;
			if(bytes_do <= 0) break;
			
			write_header(tx_buffer, 0x20, sdram_addr/2, bytes_do/2);
			status = FT_Write(handle, tx_buffer, 10, &bytes_written);
			if(status == FT_OK){
				//printf("    Wrote [header] %d bytes ok\n", bytes_written);
			}else{
				fail(MODE_DUMPRAM);
			}

			status = FT_Read( handle, buffer, bytes_do, &bytes_written);
			if(status == FT_OK){
				//printf("    Read [data]   %d bytes ok\n", bytes_written);
				fwrite(buffer, bytes_do, 1, fp);
			}else{
				fail(MODE_DUMPRAM);
			}

			// check for a timeout
			if(bytes_written == 0){
				printf("\n *** Timed out\n");
				printf(" *** Please cycle power and try again\n");
				fail(ERR_TIMEOUT);
			}

			bytes_left -= bytes_written;
			bytes_done += bytes_written;
			sdram_addr += bytes_written;

			// progress bar
			blocks_done = ((float)bytes_done / file_size)*64;
			blocks_left = 64-blocks_done;
			printf("\r   ");
			for(i = 0; i < blocks_done; i++) printf("\xdb");
			for(i = 0; i < blocks_left; i++) printf("\xb0");
		}

		// erase progress bar
		printf("\r");	for(i = 0; i < 79; i++) printf(" "); printf("\r");		
	}

	printf("\n - Done\n");

	fclose(fp);
	FT_Close(handle);

	return 0;
}

void fail(FT_STATUS st)
{ 
	if(st == FT_OK) return;

	printf("\n *** \n ***  FAIL: %d\n ***\n", st);
	exit(-1);
}

void load_image(char *filename)
{
	// code courtesy pinchy

	static TCHAR szFilter[] = TEXT ("Images (*.z64, *.bin, *.rom,)\0*.bin;*.z64;*.rom;\0All Files\0*.*;\0\0") ;
	static TCHAR szFileName [MAX_PATH], szTitleName [MAX_PATH] ;
	static OPENFILENAME ofn ;
	memset(&ofn,0,sizeof (OPENFILENAME));

	ofn.lStructSize       = sizeof (OPENFILENAME) ;
	ofn.lpstrFilter       = szFilter ;
	ofn.lpstrFile         = szFileName ;
	ofn.nMaxFile          = MAX_PATH ;
	ofn.lpstrFileTitle    = szTitleName ;
	ofn.nMaxFileTitle     = MAX_PATH ;
	ofn.lpfnHook		  = NULL;//OFNHookProc;
	ofn.Flags			  = OFN_ENABLEHOOK|OFN_EXPLORER|OFN_ENABLESIZING;
//	ofn.lpstrInitialDir   = ".";

	GetOpenFileName (&ofn) ;
	if(ofn.lpstrFile[0] != 0){
		if(filename != 0)
			strcpy(filename, ofn.lpstrFile);
	}
}