﻿using System;
using System.Collections.Generic;
using System.Text;

namespace everdrive3x.Rom
{
    class SnesRom : RomInterface
    {
        public const String ROM_TYPE_HI = "HIROM";
        public const String ROM_TYPE_LO = "LOROM";
        byte[] bin_data;
        byte[] config;
        String rom_type;
        String rom_name;
        int declarated_size = 0;
        String platform;
        String rom_region = "";

        public SnesRom(byte[] src, int offset, int len)
        {
            int region = 0;
            if (len < 131072)
            {
                throw new Exception("non SNES rom format");
            }
            bin_data = new byte[len];
            config = new byte[1];
            System.Array.Copy(src, offset, bin_data, 0, len);

            if (isSnesCrc(bin_data, 32704 + 28))
            {
                rom_type = ROM_TYPE_LO;
                rom_name = Misc.DataTool.getHeaderString(bin_data, 32768 - 64, 21);
                config[0] = (byte)'L';
                declarated_size = bin_data[32768 - 41];
                region = bin_data[32768 - 39];
            }

            else if (isSnesCrc(bin_data, 32704 + 28 + 512))
            {
                rom_type = ROM_TYPE_LO;
                bin_data = new byte[len - 512];
                System.Array.Copy(src, offset + 512, bin_data, 0, len - 512);
                config[0] = (byte)'L';
                rom_name = Misc.DataTool.getHeaderString(bin_data, 32768 - 64, 21);
                declarated_size = bin_data[32768 - 41];
                region = bin_data[32768 - 39];
            }

            else if (isSnesCrc(bin_data, 65472 + 28))
            {
                rom_type = ROM_TYPE_HI;
                config[0] = (byte)'H';
                rom_name = Misc.DataTool.getHeaderString(bin_data, 65536 - 64, 21);
                declarated_size = bin_data[65536 - 41];
                region = bin_data[65536 - 39];
            }

            else if (isSnesCrc(bin_data, 65472 + 28 + 512))
            {
                rom_type = ROM_TYPE_HI;
                bin_data = new byte[len - 512];
                System.Array.Copy(src, offset + 512, bin_data, 0, len - 512);
                rom_name = Misc.DataTool.getHeaderString(bin_data, 65536 - 64, 21);
                config[0] = (byte)'H';
                declarated_size = bin_data[65536 - 41];
                region = bin_data[65536 - 39];
            }
            else throw new Exception("non SNES rom format");

            if (declarated_size == 0x08)
            {
                declarated_size = 256 * 1024;
            }
            else if (declarated_size == 0x09)
            {
                declarated_size = 512 * 1024;
            }
            else if (declarated_size == 0x0a)
            {
                declarated_size = 1024 * 1024;
            }
            else if (declarated_size == 0x0b)
            {
                declarated_size = 2048 * 1024;
            }
            else if (declarated_size == 0x0c)
            {
                declarated_size = 4096 * 1024;
            }
            else declarated_size = 0;

            if (region == 0 || region == 1 || region == 0x0d)
            {
                rom_region = "NTSC";
            }
            else if (region >= 0x02 && region <= 0x0c)
            {
                rom_region = "PAL";
            }
            else rom_region = "unknown";

            platform = Hardware.PlatformList.NAME_SNES;

            if (rom_type == ROM_TYPE_LO)
            {
                byte [] buff = new byte[bin_data.Length * 2];
                for (int i = 0; i < bin_data.Length; i += 32768)
                {
                    System.Array.Copy(bin_data, i, buff, i * 2, 32768);
                }
                bin_data = buff;
            }
        }

        bool isSnesCrc(byte[] data, int offset)
        {
            UInt16 crc_a = (UInt16)((data[offset++] & 0xff) << 8);
            crc_a |= (UInt16)(data[offset++] & 0xFF);
            UInt16 crc_b = (UInt16)((data[offset++] & 0xff) << 8);
            crc_b |= (UInt16)(data[offset++] & 0xFF);
            crc_b ^= 0xffff;
            if (crc_a == 0 || crc_a == 0xffff || crc_b == 0 || crc_b == 0xffff)
            {
                return false;
            }
            if (crc_a == crc_b)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        public byte[] BinData
        {
            get { return bin_data; }
        }

        public byte[] Config
        {
            get { return config; }
        }

        public String Name
        {
            get { return rom_name; }
        }

        public String Platform
        {
            get { return platform; }
        }



        public String RomType
        {
            get { return rom_type; }
        }

        public int DeclaratedSize
        {
            get { return declarated_size; }
        }
        public String Region
        {
            get { return rom_region; }
            set { }
        }
    }
}
